/*
	AHid.h

	Interface of AHid.dll v0.1.2
	This driver can be used to read and write to custom USB HID devices.

	Copyright 18.06.2024 by Ralf Handrich
*/
#ifndef __AHID_H
#define __AHID_H

#include <Windows.h>

#ifndef AHIDLIBAPI
#define AHIDLIBAPI extern "C" __declspec(dllimport)
#endif //AHIDLIBAPI

/*
	Report types
*/
#define AHID_REPORT_TYPE_INPUT (0)
#define AHID_REPORT_TYPE_OUTPUT (1)

/*
	Function return values
*/
#define AHID_OK (0)
#define AHID_NOT_INITIALIZED (1)
#define AHID_INVALID_PARAMETER (2)
#define AHID_NO_DATA (4)
#define AHID_NOT_CONNECTED (5)
#define AHID_ERROR (-1)

/*
	Function prototypes
*/
/*
	int AHid_init( unsigned char* pucParm )

	Initialization of the DLL. This function must be called before any other.

	Parameters:		pucParm				- Additional parameter ( can be NULL ).

	Return value:	AHID_OK				- Always.
*/
AHIDLIBAPI int AHid_init( unsigned char* pucParm );

/*
	int AHid_register( int* phPipe, int xVid, int xPid, int xIntfId, int xRepId, int xRepSize, int xRepType )

	Create a pipe handle that is used for writing or reading data to/from your HID device. The parameters used are all device
	specific. Therefore, you need access to the USB descriptors of your device.

	Parameters:		phPipe		- A pipe handle of a report that is used in other functions.
					xVid		- Vendor ID.
					xPid		- Product ID.
					xIntfId		- Interface ID ( set it to -1 in case no interfaces are defined in your HID device ).
					xRepId		- Report ID ( set it to 0 in case no ID is defined in your report ).
					xRepSize	- Transfer size defined in the HID report.
					xRepType	- Report types (Input or Output).

	Return value:	AHID_NOT_INITIALIZED	- AHid.dll is not initialized. Call AHid_init() first.
					AHID_INVALID_PARAMETER	- VID, PID, Report ID, Size or Type is invalid.
					AHID_OK					- Registration done.
*/
AHIDLIBAPI int AHid_register( int* phPipe, int xVid, int xPid, int xIntfId, int xRepId, int xRepSize, int xRepType );

/*
	int AHid_deregister( int* phPipe )

	Remove a pipe handle.

	Parameters:		phPipe		- The pipe handle to be remove. It cannot be used any longer in other functions.

	Return value:	AHID_OK		- Always.
*/
AHIDLIBAPI int AHid_deregister( int* phPipe );

/*
	int AHid_write( int hPipe, unsigned char * pucBuffer, int xBytesToWrite, int* pxBytesWritten )

	Write data to your HID device.

	Parameters:		hPipe			- The pipe handle of an Output report.
					pucBuffer		- A buffer with the data to write.
					xBytesToWrite	- The amount of data to write.
					pxBytesWritten	- The number of bytes written.

	Return value:	AHID_INVALID_PARAMETER	- Pipe handle invalid.
					AHID_ERROR				- An unspecified error occured.
					AHID_OK					- Data written to device.

*/
AHIDLIBAPI int AHid_write( int hPipe, unsigned char * pucBuffer, int xBytesToWrite, int* pxBytesWritten );

/*
	int AHid_read( int hPipe, unsigned char * pucBuffer, int xBytesToRead, int* pxBytesRead )

	Read data from your HID device.

	Parameters:		hPipe			- The pipe handle of an Input report.
					pucBuffer		- A buffer with the data read.
					xBytesToRead	- The amount of data to read.
					pxBytesRead		- The number of bytes read.

	Return value:	AHID_INVALID_PARAMETER	- Pipe handle invalid.
					AHID_ERROR				- An unspecified error occured.
					AHID_NO_DATA			- No data received.
					AHID_OK					- Data read from device.
*/
AHIDLIBAPI int AHid_read( int hPipe, unsigned char * pucBuffer, int xBytesToRead, int* pxBytesRead );

/*
	int AHid_find( int hPipe )

	Find your HID device on USB.

	Parameters:		hPipe			- The pipe handle of a report.

	Return value:	AHID_INVALID_PARAMETER	- Pipe handle invalid.
					AHID_NOT_CONNECTED		- Device not connected to USB.
					AHID_OK					- Device connected to USB.
*/
AHIDLIBAPI int AHid_find( int hPipe );

/*
	int AHid_identify( int hPipe, unsigned char * pucBuffer, int xBufferSize, int * pxBytesProcessed )

	The buffer holds a string with the GUID of your HID device. Pipe handles of the same HID device have the same GUID.

	Parameters:		hPipe				- The pipe handle of a report.
					pucBuffer			- The buffer with the data read.
					xBufferSize			- Size of the buffer.
					pxBytesProcessed	- The number of valid bytes in the buffer.

	Return value:	AHID_INVALID_PARAMETER	- Pipe handle or buffer size invalid.
					AHID_OK					- Device GUID successfully read.
*/
AHIDLIBAPI int AHid_identify( int hPipe, unsigned char * pucBuffer, int xBufferSize, int * pxBytesProcessed );

/*
	int AHid_info( int * ulMajorVersion, int * ulMinorVersion, int * ulReleaseNumber )

	You can call this function to retrieve the version number of the driver.

	Parameters:		ulMajorVersion	- Major Version number.
					ulMinorVersion	- Minor Version number.
					ulReleaseNumber - Release number.

	Return value:	AHID_OK					- Always.
*/
AHIDLIBAPI int AHid_info( int* ulMajorVersion, int* ulMinorVersion, int* ulReleaseNumber);

#endif //__AHID_H


